/** @file   npccontroller.h
 * @brief   Declaration of NpcController - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_NPCCONTROLLER_H
#define H_WWW_NPCCONTROLLER_H

#include "waypointcontroller.h"

namespace WeWantWar {

/** @class  NpcController
 * @brief   Provides the default functionality that controls the NpcCharacters
 * @author  Tomi Lamminsaari
 *
 * This class provides the following behaviour by default:
 *  - Halted
 *  - NpcCharacter will follow another character. If the character has a weapon
 *    and there are aliens in sight, it shoots them.
 *  - NpcCharacter will move randomly.
 *  - NpcCharacter will follow a route.
 *  - NpcCharacter will stay put but shoots when aliens are in sight.
 */
class NpcController : public WaypointController
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** A datatype for state of this controller. */
  typedef int NpcState;
  
  /** A state that tells the object to be frozen. Then we don't react anything */
  static const int NPC_FREEZE = 0;
  /** A state when we make the object to follow another object. */
  static const int NPC_FOLLOW_OBJECT = 1;
  /** A state that makes the object move randomly. */
  static const int NPC_RANDOM_MOVEMENT = 2;
  /** A state that makes the object to follow a certain Route */
  static const int NPC_ROUTE = 3;
  /** A state that makes the object to aim and shoot the aliens. */
  static const int NPC_FIGHT = 4;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     pO                Pointer to GameObject we're supposed to
   *                              control.
   */
	NpcController( GameObject* pO );


	/** Destructor
   */
	virtual ~NpcController();

private:

	/** Copy constructor.
   * @param     rO                Reference to another NpcController
   */
  NpcController( const NpcController& rO );

	/** Assignment operator
   * @param     rO                Reference to another NpcController
   * @return    Reference to us.
   */
  NpcController& operator = ( const NpcController& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this controller
   */
  virtual void update();
  
  /** Sets the state.
   * @param     nstate            New state
   */
  virtual void state( NpcState nstate );
  
  /** Sets the distance, how close to the leader-object we try to stay when
   * in follow-mode.
   * @param     followDistance    The distance in pixels
   */
  virtual void setFollowDistance( float followDistance );


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the current state.
   * @return    Current state
   */
  NpcState state() const;


protected:

  /** The child-classes have to implement this. This method gets called by
   * from the update() - method of this class. This method gets called
   * after we've set the controller-flags so this method can overide
   * the controller commands.
   */
  virtual void updateNpc() = 0;
  
  /** Makes our object to follow another object. Pointer to object we should
   * follow must be set in m_pFollowObject - member.
   */
  virtual void followObject();
  
  /** Makes the object to follow the route.
   */
  virtual void followRoute();
  
  /** Stand still and shoots the aliens if they're close enough.
   */
  virtual void fight();
  
  /** Scans if there are any aliens in sight.
   * @return    true if there are aliens in sight.
   */
  virtual bool scanForAliens();
  
  
  ///
  /// Members
  /// =======

  /** The state this controller is in. */
  NpcState    m_npcState;
  /** Pointer to object that we're supposed to follow. */
  GameObject* m_pFollowObject;
  /** How close to the leader-object we should try to stay. */
  float       m_followDistance;
  /** The last known position the leader-object was seen */
  eng2d::Vec2D  m_lastKnownPos;
  /** A flag that indicates is the data in m_lastKnownPos - member valid. */
  bool          m_shouldFollow;
  
  /** A flag that indicates that there are aliens in sight. */
  bool    m_aliensInSight;
  /** Time until we scan again for the aliens. */
  int     m_scanDelay;

  /** Pointer to alien we should shoot */
  GameObject* m_pAlienToShoot;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: npccontroller.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:06+02  lamminsa
 * Initial revision
 *
 */
 
